package gov.va.genisis2.ms.service.impl;

import javax.transaction.Transactional;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataIntegrityViolationException;
import org.springframework.stereotype.Service;

import gov.va.genisis2.ms.converter.ConceptMappingConverter;
import gov.va.genisis2.ms.data.model.ConceptMapping;
import gov.va.genisis2.ms.data.repository.ConceptMappingRepository;
import gov.va.genisis2.ms.service.IConceptMappingService;
import gov.va.genisis2.ts.common.dto.ConceptMappingDTO;
import gov.va.genisis2.ts.common.enums.ErrorEnum;
import gov.va.genisis2.ts.common.exception.TSDuplicateDataException;
import gov.va.genisis2.ts.common.exception.TSInernalSystemException;
import gov.va.genisis2.ts.common.exception.TSRuntimeException;

@Service
public class ConceptMappingService implements IConceptMappingService {

	private static final Logger LOGGER = LogManager.getLogger(ConceptMappingService.class);

	@Autowired
	private ConceptMappingRepository conceptMappingRepository;

	@Autowired
	private ConceptMappingConverter conceptMappingConverter;

	@Override
	@Transactional
	public ConceptMappingDTO getConceptMappingByConceptUri(String conceptUri) {
		ConceptMappingDTO conceptMappingDto = null;

		try {
			ConceptMapping conceptMapping = conceptMappingRepository.findConceptMappingForConceptUri(conceptUri);

			if (null != conceptMapping) {
				conceptMappingDto = conceptMappingConverter.convert(conceptMapping);
			}
		} catch (Exception e) {
			LOGGER.error("Error in getConceptMappingByConceptUri", e);
			throw new TSRuntimeException("Error in getConceptMappingByConceptUri");
		}

		return conceptMappingDto;
	}

	@Override
	@Transactional
	public ConceptMappingDTO createConceptMapping(ConceptMappingDTO conceptMappingDto, ConceptMapping existingConceptMapping) {
		ConceptMappingDTO persistedConceptMappingDto = null;
		ConceptMapping conceptMapping = null;
		try {
			conceptMapping = conceptMappingConverter.convert(conceptMappingDto, existingConceptMapping);
			conceptMapping = conceptMappingRepository.save(conceptMapping);

			if (null != conceptMapping) {
				persistedConceptMappingDto = conceptMappingConverter.convert(conceptMapping);
			}
		} catch (DataIntegrityViolationException ex) {
			LOGGER.error(String.format(ErrorEnum.CONCEPT_MAPPING_DUPLICATE_ERROR.getErrorMessage(), conceptMappingDto.getConceptUri()), ex);
			throw new TSDuplicateDataException(
					String.format(ErrorEnum.CONCEPT_MAPPING_DUPLICATE_ERROR.getErrorMessage(), conceptMappingDto.getConceptUri()), ex);
		} catch (Exception e) {
			LOGGER.error("Error in createConceptMapping", e);
			throw new TSRuntimeException("Error in createConceptMapping");
		}

		return persistedConceptMappingDto;
	}

	@Override
	@Transactional
	public ConceptMapping removeConceptMapping(ConceptMappingDTO conceptMappingDto) {
		ConceptMapping conceptMapping = null;

		try {
			String conceptUri = conceptMappingDto.getConceptUri();

			if (null == conceptUri) {
				LOGGER.error("Concept-uri is null");
				throw new TSInernalSystemException("Concept-uri is null");
			} else {
				conceptMapping = conceptMappingRepository.findConceptMappingForConceptUri(conceptUri);

				if (null != conceptMapping) {
					conceptMappingRepository.delete(conceptMapping);
				}
			}
		} catch (Exception e) {
			LOGGER.error("Error in removeConceptMapping", e);
			throw new TSRuntimeException("Error in removeConceptMapping");
		}

		return conceptMapping;
	}
}
